#!/bin/bash
# ============================================================
# install_linux.sh
# Script de configuración completa para Linux
# Sistema Robot de Grabación Médica PACS/DICOM
# Compatible: Ubuntu 20.04+, Debian 10+, CentOS/RHEL 7+
# ============================================================
set -e

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
PROJECT_DIR="$(dirname "$SCRIPT_DIR")"
HELPER_SCRIPT="$SCRIPT_DIR/cdrom_helper.sh"

echo "======================================================="
echo "  Sistema Robot de Grabación Médica - Setup Linux"
echo "  Proyecto: $PROJECT_DIR"
echo "======================================================="
echo ""

# Verificar ejecución como root
if [ "$EUID" -ne 0 ]; then
    echo "⚠  Este script debe ejecutarse como root:"
    echo "   sudo bash $0"
    exit 1
fi

# ── Detectar gestor de paquetes ─────────────────────────────
if command -v apt-get &>/dev/null; then
    PKG_MGR="apt"
elif command -v dnf &>/dev/null; then
    PKG_MGR="dnf"
elif command -v yum &>/dev/null; then
    PKG_MGR="yum"
else
    echo "ERROR: Gestor de paquetes no reconocido."
    exit 1
fi

echo "[1/6] Actualizando repositorios..."
case $PKG_MGR in
    apt) apt-get update -qq ;;
    dnf) dnf makecache -q   ;;
    yum) yum makecache -q   ;;
esac

echo ""
echo "[2/6] Instalando PHP y extensiones..."
case $PKG_MGR in
    apt) apt-get install -y php php-cli php-json php-mbstring ;;
    dnf) dnf install -y php php-cli php-json php-mbstring ;;
    yum) yum install -y php php-cli php-json php-mbstring ;;
esac

echo ""
echo "[3/6] Instalando herramientas de grabación CD/DVD..."
case $PKG_MGR in
    apt)
        apt-get install -y wodim genisoimage growisofs eject cdrkit-compat 2>/dev/null || \
        apt-get install -y wodim genisoimage eject
        ;;
    dnf)
        dnf install -y wodim genisoimage dvd+rw-tools eject 2>/dev/null || \
        dnf install -y cdrecord genisoimage dvd+rw-tools
        ;;
    yum)
        yum install -y wodim genisoimage dvd+rw-tools eject 2>/dev/null || \
        yum install -y cdrecord genisoimage dvd+rw-tools
        ;;
esac

echo ""
echo "[4/6] Configurando permisos del proyecto..."

# Detectar usuario del servidor web
WEB_USER="www-data"
if id "apache"   &>/dev/null; then WEB_USER="apache"; fi
if id "httpd"    &>/dev/null; then WEB_USER="httpd";  fi
if id "www-data" &>/dev/null; then WEB_USER="www-data"; fi
echo "  Usuario del servidor web detectado: $WEB_USER"

# Crear carpetas del proyecto con permisos correctos
for DIR in estudios_pendientes estudios_procesados estudios_quemados backend frontend visor; do
    if [ -d "$PROJECT_DIR/$DIR" ]; then
        chown -R "$WEB_USER:$WEB_USER" "$PROJECT_DIR/$DIR"
        chmod -R 775 "$PROJECT_DIR/$DIR"
        echo "  ✓ Permisos aplicados a $DIR"
    fi
done

# db.json escribible por el servidor web
touch "$PROJECT_DIR/backend/db.json"
chown "$WEB_USER:$WEB_USER" "$PROJECT_DIR/backend/db.json"
chmod 664 "$PROJECT_DIR/backend/db.json"

# Permisos de dispositivos ópticos (y regla udev para persistencia)
echo "  Configurando acceso a dispositivos ópticos..."
UDEV_FILE="/etc/udev/rules.d/99-cdrom-robot.rules"
cat > "$UDEV_FILE" << UDEV_EOF
# Permisos para el Robot de Grabación Médica
KERNEL=="sr*", GROUP="cdrom", MODE="0666"
KERNEL=="scd*", GROUP="cdrom", MODE="0666"
UDEV_EOF

# Aplicar permisos inmediatamente
for DEV in /dev/sr* /dev/scd* /dev/cdrom /dev/dvd /dev/dvdrw; do
    if [ -e "$DEV" ]; then
        chmod 666 "$DEV"
        chown root:cdrom "$DEV" 2>/dev/null || true
        echo "  ✓ Permisos aplicados a $DEV"
    fi
done

# Recargar reglas udev
udevadm control --reload-rules && udevadm trigger 2>/dev/null || true

# Agregar el usuario web al grupo cdrom/optical/disk
for GRP in cdrom optical disk; do
    if getent group "$GRP" &>/dev/null; then
        usermod -aG "$GRP" "$WEB_USER" 2>/dev/null || true
        echo "  ✓ $WEB_USER agregado al grupo $GRP"
    fi
done

echo ""
echo "[5/6] Configurando sudo para operaciones de CD-ROM sin contraseña..."

# Copiar y preparar el script helper
chmod +x "$HELPER_SCRIPT"
chown root:root "$HELPER_SCRIPT"

# Detectar ruta absoluta del eject y wodim para sudoers
EJECT_PATH=$(command -v eject 2>/dev/null || echo "/usr/bin/eject")
WODIM_PATH=$(command -v wodim 2>/dev/null || echo "/usr/bin/wodim")
GROWISOFS_PATH=$(command -v growisofs 2>/dev/null || echo "/usr/bin/growisofs")
GENISOIMAGE_PATH=$(command -v genisoimage 2>/dev/null || echo "/usr/bin/genisoimage")
DD_PATH=$(command -v dd 2>/dev/null || echo "/bin/dd")
BLKID_PATH=$(command -v blkid 2>/dev/null || echo "/sbin/blkid")

# Crear archivo sudoers específico para este sistema
SUDOERS_FILE="/etc/sudoers.d/robot-medico-cdrom"
cat > "$SUDOERS_FILE" << SUDOERS_EOF
# Robot de Grabación Médica - Permisos CD-ROM
# Generado automáticamente por install_linux.sh
# Permite al servidor web ejecutar operaciones de disco sin contraseña

# Helper principal del sistema
$WEB_USER ALL=(root) NOPASSWD: $HELPER_SCRIPT *

# Comandos individuales de CD-ROM (con y sin argumentos)
$WEB_USER ALL=(root) NOPASSWD: $EJECT_PATH *
$WEB_USER ALL=(root) NOPASSWD: $WODIM_PATH *
$WEB_USER ALL=(root) NOPASSWD: $GROWISOFS_PATH *
$WEB_USER ALL=(root) NOPASSWD: $GENISOIMAGE_PATH *
$WEB_USER ALL=(root) NOPASSWD: $DD_PATH if=/dev/sr* *
$WEB_USER ALL=(root) NOPASSWD: $DD_PATH if=/dev/scd* *
$WEB_USER ALL=(root) NOPASSWD: $BLKID_PATH *

# Permitir control total sobre el comando eject para el usuario web
$WEB_USER ALL=(root) NOPASSWD: /usr/bin/eject *
$WEB_USER ALL=(root) NOPASSWD: /bin/eject *

# Permisos en tiempo de ejecución
$WEB_USER ALL=(root) NOPASSWD: /bin/chmod 666 /dev/sr*
$WEB_USER ALL=(root) NOPASSWD: /bin/chmod 666 /dev/scd*
$WEB_USER ALL=(root) NOPASSWD: /usr/bin/chmod 666 /dev/sr*
$WEB_USER ALL=(root) NOPASSWD: /usr/bin/chmod 666 /dev/scd*
SUDOERS_EOF

# Asegurar permisos correctos del archivo sudoers
chmod 440 "$SUDOERS_FILE"
chown root:root "$SUDOERS_FILE"

# Validar sintaxis del sudoers
if visudo -c -f "$SUDOERS_FILE" &>/dev/null; then
    echo "  ✓ Archivo sudoers creado y validado: $SUDOERS_FILE"
else
    echo "  ✗ Error en sudoers, eliminando..."
    rm -f "$SUDOERS_FILE"
    exit 1
fi

echo ""
echo "[6/6] Verificando instalación..."
echo ""

check_cmd() {
    if command -v "$1" &>/dev/null; then
        echo "  ✓ $1 → $(command -v $1)"
    else
        echo "  ✗ $1 → NO INSTALADO"
    fi
}

check_cmd php
check_cmd eject
check_cmd wodim
check_cmd cdrecord
check_cmd genisoimage
check_cmd growisofs
check_cmd blkid
check_cmd dd

echo ""
echo "── Dispositivos ópticos ─────────────────────────────"
ls /dev/sr* /dev/scd* /dev/cdrom /dev/dvd 2>/dev/null | while read d; do
    echo "  → $d (permisos: $(stat -c %A $d 2>/dev/null))"
done || echo "  (Ningún dispositivo detectado en este momento)"

echo ""
echo "── Prueba de sudo para $WEB_USER ───────────────────"
sudo -u "$WEB_USER" sudo "$HELPER_SCRIPT" test 2>&1 | head -3

echo ""
echo "── Versión PHP ──────────────────────────────────────"
php --version | head -1

echo ""
echo "======================================================="
echo "  ✅ Instalación completada exitosamente."
echo ""
echo "  Sudoers configurado: $SUDOERS_FILE"
echo "  Usuario web: $WEB_USER"
echo ""
echo "  Para INICIAR el robot de grabación:"
echo "    cd $PROJECT_DIR/backend && php robot.php"
echo ""
echo "  Para VERIFICAR que el botón expulsar funciona:"
echo "    sudo -u $WEB_USER sudo $HELPER_SCRIPT eject /dev/sr0"
echo ""
echo "  URL del dashboard:"
echo "    http://localhost/SystemRobotWeb/frontend/dist/"
echo "======================================================="
