import React, { useState, useEffect, useRef, useCallback } from 'react';

console.log("Dashboard Robot Médico V3.0 - Tiempo Real Activo");

// Detección automática de la raíz del proyecto
const getApiUrl = (route = '') => {
  // En desarrollo con Vite (puerto 5173), apuntar al localhost de XAMPP
  if (window.location.port === '5173') {
    const base = `http://localhost/SystemRobotWeb/backend/api.php`;
    return route ? `${base}/${route}?route=${route}` : base;
  }

  /**
   * En producción (XAMPP), usamos una ruta relativa.
   * Si la página es: http://192.168.0.207/SystemRobotWeb/index.html
   * La API está en: http://192.168.0.207/SystemRobotWeb/backend/api.php
   */
  let path = window.location.pathname;

  // Si estamos en la raíz o en index.html, buscamos la carpeta /backend/
  // Quitamos el nombre del archivo si existe (ej: index.html)
  if (path.includes('.')) {
    path = path.substring(0, path.lastIndexOf('/') + 1);
  } else if (!path.endsWith('/')) {
    path += '/';
  }

  const baseUrl = `${path}backend/api.php`;
  return route ? `${baseUrl}/${route}?route=${route}` : baseUrl;
};

const API_BASE = getApiUrl();

const BADGE_LABELS = {
  pendiente: { label: '⏳ PENDIENTE', cls: 'badge-pendiente' },
  preparando: { label: '⚙️ PREPARANDO', cls: 'badge-preparando' },
  esperando_disco: { label: '💿 ESP. DISCO', cls: 'badge-esperando' },
  grabando: { label: '🔴 GRABANDO', cls: 'badge-grabando' },
  finalizado: { label: '✅ FINALIZADO', cls: 'badge-finalizado' },
  error: { label: '❌ ERROR', cls: 'badge-error' },
};

function useInterval(callback, delay) {
  const savedCallback = useRef(callback);
  useEffect(() => { savedCallback.current = callback; }, [callback]);
  useEffect(() => {
    if (delay === null) return;
    const id = setInterval(() => savedCallback.current(), delay);
    return () => clearInterval(id);
  }, [delay]);
}

function App() {
  const [estudios, setEstudios] = useState([]);
  const [stats, setStats] = useState({ grabados_hoy: 0, pendientes: 0, errores: 0 });
  const [unidades, setUnidades] = useState([]);
  const [loading, setLoading] = useState({});
  const [ejecting, setEjecting] = useState({});
  const [selectedUnit, setSelectedUnit] = useState('');
  const [connError, setConnError] = useState(null);
  const [statusMsg, setStatusMsg] = useState(null);
  const [statusOk, setStatusOk] = useState(true);
  const [lastUpdated, setLastUpdated] = useState('--:--:--');
  const [isLive, setIsLive] = useState(false);
  const [pollCount, setPollCount] = useState(0);
  const [isSettingsOpen, setIsSettingsOpen] = useState(false);
  const [config, setConfig] = useState({
    modo_automatico: '1',
    velocidad_grabacion: '8x',
    unidad_optica: '/dev/sr0',
    expulsar_al_terminar: '1'
  });
  const [license, setLicense] = useState({ activada: false, vencimiento: '', valida: true });
  const [activationData, setActivationData] = useState({ usuario: '', password: '', dias: '365', nuevo_usuario: '', nuevo_password: '' });
  const [isChangingCreds, setIsChangingCreds] = useState(false);
  const [showActivation, setShowActivation] = useState(false);
  const statusTimer = useRef(null);

  const showStatus = useCallback((msg, ok = true, duration = 4000) => {
    if (statusTimer.current) clearTimeout(statusTimer.current);
    setStatusMsg(msg);
    setStatusOk(ok);
    if (duration > 0) {
      statusTimer.current = setTimeout(() => setStatusMsg(null), duration);
    }
  }, []);

  const fetchLicense = useCallback(async () => {
    try {
      const res = await fetch(getApiUrl('licencia'));
      if (res.ok) {
        const data = await res.json();
        setLicense({ ...data.licencia, valida: data.valida });
      }
    } catch (err) { console.error("Error al cargar licencia:", err); }
  }, []);

  const fetchConfig = useCallback(async () => {
    try {
      const res = await fetch(getApiUrl('configuracion'));
      if (res.ok) setConfig(await res.json());
    } catch (err) { console.error("Error al cargar config:", err); }
  }, []);

  const saveConfig = async (e) => {
    if (e) e.preventDefault();
    try {
      const res = await fetch(getApiUrl('configuracion'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(config)
      });
      if (res.ok) {
        showStatus('✅ Configuración guardada correctamente');
        setIsSettingsOpen(false);
        fetchData();
      }
    } catch (err) { showStatus('❌ Error al guardar configuración', false); }
  };

  const activarLicencia = async (e) => {
    if (e) e.preventDefault();
    try {
      const res = await fetch(getApiUrl('activar'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(activationData)
      });
      const data = await res.json();
      if (data.exito) {
        showStatus('✅ ' + data.mensaje);
        fetchLicense();
        setShowActivation(false);
      } else {
        showStatus('❌ ' + data.mensaje, false);
      }
    } catch (err) { showStatus('❌ Error de red al activar', false); }
  };

  const updateLicenseAdmin = async (newLic) => {
    try {
      const res = await fetch(getApiUrl('licencia'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(newLic)
      });
      if (res.ok) {
        showStatus('✅ Licencia actualizada (Modo Admin)');
        fetchLicense();
      }
    } catch (err) { showStatus('❌ Error al actualizar licencia', false); }
  };

  const fetchData = useCallback(async () => {
    try {
      const [respEst, respStat, respUni] = await Promise.all([
        fetch(getApiUrl('estudios')),
        fetch(getApiUrl('estadisticas')),
        fetch(getApiUrl('unidades')),
      ]);

      if (!respEst.ok) throw new Error(`HTTP ${respEst.status}`);
      const estudiosData = await respEst.json();
      setEstudios(Array.isArray(estudiosData) ? estudiosData : []);

      if (respStat.ok) setStats(await respStat.json());

      if (respUni.ok) {
        const unidadesData = await respUni.json();
        const arr = Array.isArray(unidadesData) ? unidadesData : [];
        setUnidades(arr);
        setSelectedUnit(prev => {
          if (!prev && arr.length > 0) {
            // Si hay una unidad preferida en config y está disponible, usar esa
            const preferred = arr.find(u => u.unidad === config.unidad_optica);
            if (preferred) return preferred.unidad;

            const firstBlank = arr.find(u => u.vacio && u.disco_insertado);
            return firstBlank ? firstBlank.unidad : arr[0].unidad;
          }
          return prev;
        });
      }

      setConnError(null);
      setIsLive(true);
      setLastUpdated(new Date().toLocaleTimeString('es-ES'));
      setPollCount(c => c + 1);
    } catch (err) {
      console.error("Error de conexión:", err);
      setIsLive(false);
      setConnError(`Sin conexión con el servidor. Verifica que XAMPP esté activo. (URL: ${API_BASE})`);
    }
  }, [config.unidad_optica]);

  // Primer fetch inmediato
  useEffect(() => {
    fetchData();
    fetchConfig();
    fetchLicense();
  }, [fetchData, fetchConfig, fetchLicense]);

  // Actualizar Título y Favicon dinámicamente
  useEffect(() => {
    const nombre = config.nombre_sistema || 'Robot Médico';
    const logo = config.logo_sistema || '💿';

    // Actualizar título de la ventana
    document.title = `${logo} ${nombre}`;

    // Actualizar favicon dinámicamente (Único y escalable)
    // Primero eliminamos todos los favicons existentes para que solo quede el nuestro
    const existingIcons = document.querySelectorAll("link[rel*='icon']");
    existingIcons.forEach(el => el.parentNode.removeChild(el));

    // Crear un nuevo link de favicon único
    const newIcon = document.createElement('link');
    newIcon.rel = 'icon';
    newIcon.type = 'image/svg+xml';

    // Generamos un SVG on-the-fly con el logo
    const svgIcon = `
      <svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22>
        <text y=%22.9em%22 font-size=%2290%22>${logo}</text>
      </svg>
    `.trim();

    newIcon.href = `data:image/svg+xml,${svgIcon.replace(/"/g, "'").replace(/#/g, "%23")}`;
    document.getElementsByTagName('head')[0].appendChild(newIcon);
  }, [config.nombre_sistema, config.logo_sistema]);


  // Polling cada 1.5 segundos para respuesta casi instantánea
  useInterval(() => {
    fetchData();
    if (pollCount % 20 === 0) fetchLicense(); // Pollear licencia cada 30 segundos aprox
  }, 1500);

  const iniciarGrabacion = async (id) => {
    if (!selectedUnit) {
      showStatus('⚠️ Seleccione una unidad de grabación primero', false);
      return;
    }
    setLoading(prev => ({ ...prev, [id]: true }));
    try {
      const res = await fetch(getApiUrl('grabar'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ id, unidad: selectedUnit })
      });
      const data = await res.json();
      if (data.error) {
        showStatus(`❌ ${data.error}`, false);
      } else {
        showStatus(`🚀 Grabación iniciada para estudio #${id} en ${selectedUnit}`, true);
      }
      fetchData();
    } catch (err) {
      showStatus('❌ Error al iniciar grabación', false);
    } finally {
      setLoading(prev => ({ ...prev, [id]: false }));
    }
  };

  const reencolarEstudio = async (id) => {
    try {
      const res = await fetch(getApiUrl('estudio'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ id, estado: 'pendiente' })
      });
      if (res.ok) {
        showStatus('↔️ Estudio vuelto a estado pendiente');
        fetchData();
      }
    } catch (err) {
      showStatus('❌ Error al cambiar estado', false);
    }
  };

  const expulsarDisco = async (unidad) => {
    if (ejecting[unidad]) return;
    setEjecting(prev => ({ ...prev, [unidad]: true }));
    showStatus(`⏏ Expulsando unidad ${unidad}...`, true, 0);
    try {
      const res = await fetch(getApiUrl('expulsar'), {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ unidad })
      });
      const data = await res.json();
      if (data.exito) {
        showStatus(`✅ Unidad ${unidad} expulsada correctamente`, true, 5000);
      } else {
        showStatus(`⚠️ Comando enviado a ${unidad} (puede requerir permisos de administrador)`, false, 6000);
      }
      setTimeout(fetchData, 1500);
    } catch (err) {
      showStatus(`❌ Error al expulsar ${unidad}`, false, 5000);
    } finally {
      setEjecting(prev => ({ ...prev, [unidad]: false }));
    }
  };

  const pendientesCount = estudios.filter(e => e.estado === 'pendiente').length;
  const grabandoCount = estudios.filter(e => e.estado === 'grabando').length;

  return (
    <div className="dashboard">
      {/* HEADER */}
      <header>
        <div className="title-group">
          <div className="title-icon">{config.logo_sistema || '💿'}</div>
          <div>
            <h1>{config.nombre_sistema || 'Robot Médico'} <span className="version-tag">V3.1</span></h1>
            <p className="subtitle">Sistema de Grabación Automatizada PACS/DICOM</p>
          </div>
        </div>
        <div className="header-right">
          <div className="header-actions" style={{ display: 'flex', gap: '1rem', marginBottom: '0.5rem', justifyContent: 'flex-end' }}>
            <button className="btn-config" onClick={() => { fetchConfig(); setIsSettingsOpen(true); }}>
              ⚙️ Configuración
            </button>
            <div className={`live-indicator ${isLive ? 'live-on' : 'live-off'}`}>
              <span className="live-dot"></span>
              <span>{isLive ? 'EN VIVO' : 'SIN SEÑAL'}</span>
            </div>
          </div>
          <div className="server-info">
            <span className="server-host">🖥 {window.location.hostname}</span>
            <span className="update-time">Actualizado: {lastUpdated}</span>
          </div>
        </div>
      </header>

      {/* MODAL CONFIGURACION */}
      {isSettingsOpen && (
        <div className="modal-overlay">
          <div className="modal-content">
            <div className="modal-header">
              <h2>⚙️ Ajustes del Sistema</h2>
              <button className="btn-close" onClick={() => setIsSettingsOpen(false)}>&times;</button>
            </div>
            <form onSubmit={saveConfig}>
              <div className="modal-body">
                <div className="config-section">
                  <h3>Hardware y Grabación</h3>
                  <div className="form-group">
                    <label>💿 Unidad Grabadora Predeterminada</label>
                    <select
                      className="form-select"
                      value={config.unidad_optica}
                      onChange={e => setConfig({ ...config, unidad_optica: e.target.value })}
                    >
                      {unidades.map(u => (
                        <option key={u.unidad} value={u.unidad}>{u.unidad} ({u.tipo})</option>
                      ))}
                      {unidades.length === 0 && <option value="/dev/sr0">/dev/sr0 (Default)</option>}
                    </select>
                  </div>
                  <div className="form-group">
                    <label>⚡ Velocidad de Grabación</label>
                    <select
                      className="form-select"
                      value={config.velocidad_grabacion}
                      onChange={e => setConfig({ ...config, velocidad_grabacion: e.target.value })}
                    >
                      <option value="4x">4x (Más seguro)</option>
                      <option value="8x">8x (Recomendado)</option>
                      <option value="16x">16x (Rápido)</option>
                      <option value="MAX">Máxima permitida</option>
                    </select>
                  </div>
                </div>

                <div className="config-section">
                  <h3>Identidad del Sistema</h3>
                  <div className="form-group">
                    <label>📝 Nombre del Sistema</label>
                    <input
                      type="text"
                      className="form-input"
                      value={config.nombre_sistema || ''}
                      onChange={e => setConfig({ ...config, nombre_sistema: e.target.value })}
                      placeholder="Ej: Robot Médico San José"
                    />
                  </div>
                  <div className="form-group">
                    <label>🖼️ Logo o Icono (Emoji o Texto)</label>
                    <input
                      type="text"
                      className="form-input"
                      value={config.logo_sistema || ''}
                      onChange={e => setConfig({ ...config, logo_sistema: e.target.value })}
                      placeholder="Ej: 💿 o 🏥"
                    />
                  </div>
                </div>


                <div className="config-section">
                  <h3>Automatización</h3>
                  <div className="toggle-group">
                    <div className="toggle-info">
                      <h4>Modo Automático</h4>
                      <p>Graba los estudios apenas se detectan</p>
                    </div>
                    <label className="switch">
                      <input
                        type="checkbox"
                        checked={config.modo_automatico === '1'}
                        onChange={e => setConfig({ ...config, modo_automatico: e.target.checked ? '1' : '0' })}
                      />
                      <span className="slider"></span>
                    </label>
                  </div>
                  <div className="toggle-group">
                    <div className="toggle-info">
                      <h4>Expulsar al terminar</h4>
                      <p>Abre la bandeja automáticamente tras grabar</p>
                    </div>
                    <label className="switch">
                      <input
                        type="checkbox"
                        checked={config.expulsar_al_terminar === '1'}
                        onChange={e => setConfig({ ...config, expulsar_al_terminar: e.target.checked ? '1' : '0' })}
                      />
                      <span className="slider"></span>
                    </label>
                  </div>
                </div>

                <div className="config-section">
                  <h3>Licencia del Sistema</h3>
                  <div className="form-group">
                    <label>📅 Tiempo de Vigencia (Vencimiento)</label>
                    <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', background: '#f3f4f6', padding: '0.75rem', borderRadius: '8px', marginTop: '0.5rem' }}>
                      <span style={{ fontWeight: 'bold', color: license.valida ? '#059669' : '#dc2626' }}>
                        {license.vencimiento || 'No activada'} {license.valida ? '(Activa)' : '(Vencida)'}
                      </span>
                      <button
                        type="button"
                        className="btn btn-primary"
                        onClick={() => { setIsSettingsOpen(false); setShowActivation(true); }}
                        style={{ padding: '0.4rem 0.8rem', fontSize: '0.85rem' }}
                      >
                        ⚡ Incrementar Licencia
                      </button>
                    </div>
                    <small className="help-text">Haga clic en incrementar para renovar o extender los días de vigencia del sistema.</small>
                  </div>
                </div>

              </div>
              <div className="modal-footer">
                <button type="button" className="btn btn-refresh" onClick={() => setIsSettingsOpen(false)}>Cancelar</button>
                <button type="submit" className="btn btn-primary">Guardar Ajustes</button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* MODAL ACTIVACION (Bloqueante si no es válida) */}
      {(!license.valida || showActivation) && (
        <div className="modal-overlay" style={{ zIndex: 10000 }}>
          <div className="modal-content" style={{ maxWidth: '400px' }}>
            <div className="modal-header">
              <h2>🔐 Activación de Licencia</h2>
              {license.valida && <button className="btn-close" onClick={() => setShowActivation(false)}>&times;</button>}
            </div>
            <div className="modal-body">
              {!license.valida && (
                <div className="badge badge-error" style={{ width: '100%', marginBottom: '1rem', textAlign: 'center', padding: '1rem' }}>
                  LA LICENCIA HA VENCIDO O NO ESTÁ ACTIVA
                </div>
              )}
              <p style={{ marginBottom: '1rem', fontSize: '0.9rem' }}>Ingrese sus credenciales y clave de activación para continuar.</p>
              <form onSubmit={activarLicencia}>
                <div className="form-group">
                  <label>Usuario</label>
                  <input
                    type="text"
                    className="form-input"
                    placeholder="Ej: admin"
                    value={activationData.usuario}
                    onChange={e => setActivationData({ ...activationData, usuario: e.target.value })}
                    required
                  />
                </div>
                <div className="form-group">
                  <label>Contraseña</label>
                  <input
                    type="password"
                    className="form-input"
                    placeholder="••••••"
                    value={activationData.password}
                    onChange={e => setActivationData({ ...activationData, password: e.target.value })}
                    required
                  />
                </div>
                <div className="form-group">
                  <label>Días de Activación (Manual)</label>
                  <input
                    type="number"
                    className="form-input"
                    min="1"
                    max="3650"
                    value={activationData.dias}
                    onChange={e => setActivationData({ ...activationData, dias: e.target.value })}
                    required
                  />
                </div>

                <div className="form-group" style={{ marginTop: '1.5rem', borderTop: '1px solid #eee', paddingTop: '1rem' }}>
                  <label style={{ display: 'flex', alignItems: 'center', gap: '0.5rem', cursor: 'pointer' }}>
                    <input
                      type="checkbox"
                      checked={isChangingCreds}
                      onChange={e => setIsChangingCreds(e.target.checked)}
                    />
                    <span>Cambiar credenciales de Administrador</span>
                  </label>
                </div>

                {isChangingCreds && (
                  <div className="config-section" style={{
                    background: '#fffeb2',
                    padding: '1.25rem',
                    borderRadius: '12px',
                    marginTop: '1rem',
                    border: '2px solid #ecc94b',
                    boxShadow: '0 4px 6px -1px rgba(0, 0, 0, 0.1)'
                  }}>
                    <div style={{
                      display: 'flex',
                      alignItems: 'center',
                      gap: '0.75rem',
                      color: '#856404',
                      marginBottom: '1rem',
                      padding: '0.75rem',
                      background: '#fff',
                      borderRadius: '8px',
                      borderLeft: '4px solid #d69e2e'
                    }}>
                      <span style={{ fontSize: '1.5rem' }}>⚠️</span>
                      <p style={{ fontSize: '0.95rem', fontWeight: '800', margin: 0, lineHeight: '1.4' }}>
                        ¡ATENCIÓN! Se validarán sus credenciales ACTUALES (las de arriba) para poder autorizar este cambio.
                      </p>
                    </div>
                    <div className="form-group">
                      <label style={{ color: '#744210', fontWeight: 'bold' }}>Nuevo Usuario</label>
                      <input
                        type="text"
                        className="form-input"
                        style={{ borderColor: '#d69e2e' }}
                        placeholder="Ej: nuevo_admin"
                        value={activationData.nuevo_usuario}
                        onChange={e => setActivationData({ ...activationData, nuevo_usuario: e.target.value })}
                        required={isChangingCreds}
                      />
                    </div>
                    <div className="form-group">
                      <label style={{ color: '#744210', fontWeight: 'bold' }}>Nueva Contraseña</label>
                      <input
                        type="password"
                        className="form-input"
                        style={{ borderColor: '#d69e2e' }}
                        placeholder="••••••"
                        value={activationData.nuevo_password}
                        onChange={e => setActivationData({ ...activationData, nuevo_password: e.target.value })}
                        required={isChangingCreds}
                      />
                    </div>
                  </div>
                )}

                <button type="submit" className="btn btn-primary" style={{ width: '100%', marginTop: '1rem' }}>
                  🚀 Activar Sistema
                </button>
              </form>
            </div>
          </div>
        </div>
      )}

      {/* NOTIFICACIÓN FLOTANTE */}
      {statusMsg && (
        <div className={`toast-notification ${statusOk ? 'toast-ok' : 'toast-error'}`}>
          {statusMsg}
        </div>
      )}

      {/* ERROR DE CONEXIÓN */}
      {connError && (
        <div className="conn-error-banner">
          ⚠️ {connError}
        </div>
      )}

      {/* STATS */}
      <div className="stats-grid">
        <div className="stat-card stat-primary">
          <div className="stat-icon">📀</div>
          <div className="stat-info">
            <h3>Grabados Hoy</h3>
            <div className="stat-value">{stats.grabados_hoy}</div>
          </div>
        </div>
        <div className="stat-card stat-warning">
          <div className="stat-icon">⏳</div>
          <div className="stat-info">
            <h3>Pendientes</h3>
            <div className="stat-value">{stats.pendientes}</div>
          </div>
        </div>
        <div className="stat-card stat-danger">
          <div className="stat-icon">⚠️</div>
          <div className="stat-info">
            <h3>Errores</h3>
            <div className="stat-value">{stats.errores}</div>
          </div>
        </div>
        <div className="stat-card stat-success">
          <div className="stat-icon">🔴</div>
          <div className="stat-info">
            <h3>Grabando</h3>
            <div className={`stat-value ${grabandoCount > 0 ? 'stat-pulse' : ''}`}>{grabandoCount}</div>
          </div>
        </div>
      </div>

      {/* BARRA DE CONTROL */}
      <div className="control-bar">
        <label className="control-label">🎯 Unidad Activa:</label>
        <select
          className="unit-selector"
          value={selectedUnit}
          onChange={e => setSelectedUnit(e.target.value)}
        >
          {unidades.length === 0 && (
            <option value="">— Sin unidades detectadas —</option>
          )}
          {unidades.map(u => (
            <option key={u.unidad} value={u.unidad}>
              {u.unidad} &nbsp;
              {u.disco_insertado
                ? (u.vacio ? '💿 Disco Blanco ✓' : '📀 Con Datos')
                : '⬜ Vacía'}
            </option>
          ))}
        </select>
        <button
          className="btn btn-refresh"
          onClick={fetchData}
          title="Actualizar datos ahora"
        >
          🔄 Actualizar
        </button>
        <span className="poll-counter">Ciclo #{pollCount}</span>
      </div>

      {/* CONTENIDO PRINCIPAL */}
      <div className="content-grid">

        {/* TABLA DE ESTUDIOS */}
        <div className="section-card">
          <div className="section-title">
            <span>📋 Gestión de Estudios</span>
            <span className="badge-count">{estudios.length} estudios</span>
          </div>
          {estudios.length === 0 ? (
            <div className="empty-state">
              <div className="empty-icon">📭</div>
              <p>No hay estudios registrados</p>
              <small>Coloca archivos DICOM en la carpeta <code>estudios_pendientes/</code> y ejecuta el robot</small>
            </div>
          ) : (
            <div className="table-wrapper">
              <table>
                <thead>
                  <tr>
                    <th>#</th>
                    <th>Paciente</th>
                    <th>Modalidad</th>
                    <th>Detectado</th>
                    <th>Estado</th>
                    <th>Unidad</th>
                    <th>Acción</th>
                  </tr>
                </thead>
                <tbody>
                  {estudios.map(es => {
                    const b = BADGE_LABELS[es.estado] || { label: es.estado, cls: '' };
                    const esFinalizado = es.estado === 'finalizado';

                    return (
                      <tr key={es.id} className={es.estado === 'grabando' ? 'row-grabando' : ''}>
                        <td><small className="id-badge">#{es.id}</small></td>
                        <td>
                          <div style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
                            {esFinalizado && <span style={{ fontSize: '1.2rem' }} title="Procesado con éxito">✅</span>}
                            <strong>{es.paciente}</strong>
                          </div>
                        </td>
                        <td>{!esFinalizado && <span className="modalidad-tag">{es.modalidad}</span>}</td>
                        <td>
                          {!esFinalizado && (
                            <div style={{ display: 'flex', flexDirection: 'column' }}>
                              <span style={{ fontSize: '0.8rem' }}>{es.fecha}</span>
                              <small style={{ color: 'var(--text-dim)', fontSize: '0.7rem' }}>{es.hora_deteccion || '--:--'}</small>
                            </div>
                          )}
                        </td>
                        <td>
                          {!esFinalizado && (
                            <span className={`badge ${b.cls}`}>{b.label}</span>
                          )}
                        </td>
                        <td>{!esFinalizado && <small className="unidad-text">{es.unidad_asignada || '—'}</small>}</td>
                        <td>
                          {es.estado === 'pendiente' && (
                            <button
                              className="btn btn-primary btn-sm"
                              onClick={() => iniciarGrabacion(es.id)}
                              disabled={loading[es.id] || !selectedUnit}
                              title={!selectedUnit ? 'Seleccione una unidad primero' : 'Iniciar grabación'}
                            >
                              {loading[es.id] ? <span className="spinner"></span> : '▶ Grabar'}
                            </button>
                          )}

                          {(es.estado === 'finalizado' || es.estado === 'grabando' || es.estado === 'preparando' || es.estado === 'esperando_disco') && (
                            <div className="action-group" style={{ display: 'flex', gap: '0.5rem', alignItems: 'center' }}>
                              {es.estado === 'esperando_disco' && (
                                <button
                                  className="btn btn-warning btn-sm"
                                  onClick={() => reencolarEstudio(es.id)}
                                  style={{ padding: '0.2rem 0.5rem', fontSize: '0.75rem' }}
                                  title="Volver a estado pendiente"
                                >
                                  ↩ Pendiente
                                </button>
                              )}
                              {es.estado === 'grabando' && <span className="badge badge-grabando" style={{ fontSize: '0.65rem' }}>●</span>}
                              <button
                                className="btn btn-secondary btn-sm"
                                onClick={() => {
                                  // Reemplazar la ruta de la API por la del visor
                                  const visorBase = API_BASE.replace('backend/api.php', 'visor/index.html');
                                  // Pasar el ID del estudio para que el visor cargue sus datos específicos
                                  const visorUrl = `${visorBase}?id=${es.id}`;
                                  window.open(visorUrl, '_blank');
                                }}
                                title="Abrir visor de imágenes"
                              >
                                👁 Visor
                              </button>
                            </div>
                          )}

                          {es.estado === 'error' && (
                            <button
                              className="btn btn-warning btn-sm"
                              onClick={() => iniciarGrabacion(es.id)}
                              disabled={loading[es.id]}
                            >
                              🔁 Reintentar
                            </button>
                          )}
                        </td>
                      </tr>
                    );
                  })}
                </tbody>
              </table>
            </div>
          )}
        </div>

        {/* PANEL DERECHO */}
        <div className="right-panel">

          {/* ESTADO DE UNIDADES */}
          <div className="section-card">
            <div className="section-title">
              <span>💿 Unidades Ópticas</span>
              <span className="badge-count">{unidades.length}</span>
            </div>

            {unidades.length === 0 ? (
              <div className="empty-state" style={{ padding: '1rem' }}>
                <p style={{ fontSize: '0.8rem' }}>🔍 Buscando grabadoras...</p>
                <small>Asegúrate de que la grabadora esté conectada y PhP tenga permisos para ejecutar PowerShell</small>
              </div>
            ) : (
              <div className="drives-list">
                {unidades.map(u => {
                  const tieneBlanco = u.disco_insertado && u.vacio;
                  const tieneDatos = u.disco_insertado && !u.vacio;
                  const esRW = u.es_rw === true;
                  return (
                    <div key={u.unidad} className={`drive-card ${tieneBlanco ? 'drive-ready' : tieneDatos ? (esRW ? 'drive-rw' : 'drive-data') : 'drive-empty'}`}>
                      <div className="drive-header">
                        <div className="drive-letter">
                          <span className="drive-icon">
                            {tieneBlanco ? '💿' : tieneDatos ? (esRW ? '🔄' : '📀') : '⬜'}
                          </span>
                          <strong>{u.unidad}</strong>
                          {esRW && <span style={{ fontSize: '0.65rem', marginLeft: '4px', background: '#7c3aed', color: '#fff', borderRadius: '4px', padding: '1px 5px' }}>RW</span>}
                        </div>
                        <div className={`drive-status-dot ${tieneBlanco ? 'dot-green' : tieneDatos ? 'dot-yellow' : 'dot-gray'}`}></div>
                      </div>
                      <div className="drive-type">{u.tipo}</div>
                      {u.volname && <div style={{ fontSize: '0.72rem', color: '#6b7280', marginTop: '2px' }}>📝 Vol: <em>{u.volname}</em></div>}
                      <div className="drive-cap">{u.capacidad}</div>
                      <button
                        className={`btn-eject ${ejecting[u.unidad] ? 'btn-eject-loading' : ''}`}
                        onClick={() => expulsarDisco(u.unidad)}
                        disabled={ejecting[u.unidad]}
                        title={`Expulsar bandeja de ${u.unidad}`}
                      >
                        {ejecting[u.unidad]
                          ? <><span className="spinner-sm"></span> Expulsando...</>
                          : <>⏏ Expulsar {u.unidad}</>
                        }
                      </button>
                    </div>
                  );
                })}
              </div>
            )}
          </div>

          {/* MONITOR DEL SERVIDOR */}
          <div className="section-card" style={{ marginTop: '1.5rem' }}>
            <div className="section-title">📡 Monitor del Sistema</div>
            <div className="monitor-grid">
              <div className="monitor-row">
                <span className="mon-label">Estado</span>
                <span className={`mon-value ${isLive ? 'mon-ok' : 'mon-err'}`}>
                  {isLive ? '● Conectado' : '✖ Desconectado'}
                </span>
              </div>
              <div className="monitor-row">
                <span className="mon-label">Última actualización</span>
                <span className="mon-value">{lastUpdated}</span>
              </div>
              <div className="monitor-row">
                <span className="mon-label">Ciclos de polling</span>
                <span className="mon-value">#{pollCount}</span>
              </div>
              <div className="monitor-row">
                <span className="mon-label">Intervalo</span>
                <span className="mon-value">1.5 seg</span>
              </div>
              <div className="monitor-row">
                <span className="mon-label">Drives detectados</span>
                <span className="mon-value">{unidades.length}</span>
              </div>
              <div className="monitor-row">
                <span className="mon-label">Modo</span>
                <span className="mon-value mon-ok">Auto-Grabación ✓</span>
              </div>
              <div className="monitor-row">
                <span className="mon-label">Licencia</span>
                <span className={`mon-value ${license.valida ? 'mon-ok' : 'mon-err'}`}>
                  {license.valida ? 'Activa ✓' : 'Vencida ✖'}
                </span>
              </div>
              <div className="monitor-row">
                <span className="mon-label">Vencimiento</span>
                <span className="mon-value">{license.vencimiento || '—'}</span>
              </div>
              <div className="monitor-row">
                <span className="mon-label">Endpoint</span>
                <span className="mon-value mon-url">{API_BASE.replace('/backend/api.php', '')}</span>
              </div>
            </div>
            {!license.valida && (
              <button
                className="btn btn-primary"
                style={{ width: '100%', marginTop: '1rem' }}
                onClick={() => setShowActivation(true)}
              >
                🔐 Activar Licencia
              </button>
            )}
          </div>

        </div>
      </div>
    </div>
  );
}

export default App;
